<?php
require_once 'config.php';
requireDepartment(['admin']);

$error = '';
$success = '';

// Handle add/edit user
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    $username = trim($conn->real_escape_string($_POST['username']));
    $full_name = trim($conn->real_escape_string($_POST['full_name']));
    $email = trim($conn->real_escape_string($_POST['email']));
    $department = $conn->real_escape_string($_POST['department']);
    
    // Validate username
    if (empty($username) || empty($full_name) || empty($email) || empty($department)) {
        $error = 'All fields are required!';
    } else {
        // Check if username exists (for new users or when username changed)
        $check_stmt = $conn->prepare("SELECT id FROM users WHERE username = ? AND id != ?");
        $check_stmt->bind_param("si", $username, $id);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows > 0) {
            $error = 'Username already exists!';
        } else {
            if ($id > 0) {
                // Update existing user
                if (!empty($_POST['password'])) {
                    $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
                    $stmt = $conn->prepare("UPDATE users SET username = ?, password = ?, full_name = ?, email = ?, department = ? WHERE id = ?");
                    $stmt->bind_param("sssssi", $username, $password, $full_name, $email, $department, $id);
                } else {
                    $stmt = $conn->prepare("UPDATE users SET username = ?, full_name = ?, email = ?, department = ? WHERE id = ?");
                    $stmt->bind_param("ssssi", $username, $full_name, $email, $department, $id);
                }
            } else {
                // Add new user
                if (empty($_POST['password'])) {
                    $error = 'Password is required for new users!';
                } else {
                    $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
                    $stmt = $conn->prepare("INSERT INTO users (username, password, full_name, email, department) VALUES (?, ?, ?, ?, ?)");
                    $stmt->bind_param("sssss", $username, $password, $full_name, $email, $department);
                }
            }
            
            if (empty($error) && $stmt->execute()) {
                $success = $id > 0 ? 'User updated successfully!' : 'User added successfully!';
                // Reset form
                $_POST = [];
            } elseif (empty($error)) {
                $error = 'Failed to save user: ' . $conn->error;
            }
            
            if (isset($stmt)) {
                $stmt->close();
            }
        }
        $check_stmt->close();
    }
}

// Handle delete user
if (isset($_GET['delete'])) {
    $delete_id = intval($_GET['delete']);
    
    // Don't allow deleting the admin user (id=1)
    if ($delete_id != 1) {
        $delete_stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
        $delete_stmt->bind_param("i", $delete_id);
        
        if ($delete_stmt->execute()) {
            $success = 'User deleted successfully!';
        } else {
            $error = 'Failed to delete user!';
        }
        $delete_stmt->close();
    } else {
        $error = 'Cannot delete the main administrator!';
    }
}

// Get user for editing
$edit_user = null;
if (isset($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    $edit_stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
    $edit_stmt->bind_param("i", $edit_id);
    $edit_stmt->execute();
    $edit_result = $edit_stmt->get_result();
    $edit_user = $edit_result->fetch_assoc();
    $edit_stmt->close();
}

// Get all users
$users_result = $conn->query("SELECT * FROM users ORDER BY department, username");
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management | Client Management System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .users-container {
            display: grid;
            grid-template-columns: 1fr 2fr;
            gap: 2rem;
            margin-top: 2rem;
        }
        
        @media (max-width: 992px) {
            .users-container {
                grid-template-columns: 1fr;
            }
        }
        
        .form-section, .list-section {
            background: white;
            padding: 2rem;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        
        .alert {
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1.5rem;
            font-weight: 500;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .form-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .department-badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .badge-admin { background: #ffebee; color: #c62828; }
        .badge-lead { background: #e3f2fd; color: #1565c0; }
        .badge-sales { background: #e8f5e9; color: #2e7d32; }
        .badge-delivery { background: #fff3e0; color: #ef6c00; }
        .badge-project { background: #f3e5f5; color: #7b1fa2; }
        .badge-upfollow { background: #e0f7fa; color: #006064; }
        
        .user-actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn-icon {
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            text-decoration: none;
            transition: all 0.3s;
        }
        
        .btn-edit {
            background: #e3f2fd;
            color: #1976d2;
        }
        
        .btn-edit:hover {
            background: #bbdefb;
        }
        
        .btn-delete {
            background: #ffebee;
            color: #d32f2f;
        }
        
        .btn-delete:hover {
            background: #ffcdd2;
        }
        
        .password-note {
            font-size: 0.9rem;
            color: #666;
            margin-top: 0.5rem;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 10px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: bold;
            color: #667eea;
            display: block;
        }
        
        .stat-label {
            font-size: 0.9rem;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .search-box {
            margin-bottom: 1.5rem;
        }
        
        .search-box input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 1rem;
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem;
            color: #666;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: #ddd;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="main-content">
        <h1><i class="fas fa-users"></i> User Management</h1>
        
        <!-- Statistics -->
        <?php
        $total_users = $conn->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'];
        $admin_count = $conn->query("SELECT COUNT(*) as count FROM users WHERE department = 'admin'")->fetch_assoc()['count'];
        $lead_count = $conn->query("SELECT COUNT(*) as count FROM users WHERE department = 'lead'")->fetch_assoc()['count'];
        $sales_count = $conn->query("SELECT COUNT(*) as count FROM users WHERE department = 'sales'")->fetch_assoc()['count'];
        $follow_count = $conn->query("SELECT COUNT(*) as count FROM users WHERE department = 'upfollow'")->fetch_assoc()['count'];
        ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <span class="stat-value"><?php echo $total_users; ?></span>
                <span class="stat-label">Total Users</span>
            </div>
            <div class="stat-card">
                <span class="stat-value"><?php echo $admin_count; ?></span>
                <span class="stat-label">Admins</span>
            </div>
            <div class="stat-card">
                <span class="stat-value"><?php echo $lead_count; ?></span>
                <span class="stat-label">Lead Dept</span>
            </div>
            <div class="stat-card">
                <span class="stat-value"><?php echo $sales_count; ?></span>
                <span class="stat-label">Sales Dept</span>
            </div>
            <div class="stat-card">
                <span class="stat-value"><?php echo $follow_count; ?></span>
                <span class="stat-label">Up Follow</span>
            </div>
        </div>
        
        <?php if ($error): ?>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="users-container">
            <!-- User Form -->
            <div class="form-section">
                <div class="form-header">
                    <h2><?php echo $edit_user ? 'Edit User' : 'Add New User'; ?></h2>
                    <?php if ($edit_user): ?>
                        <a href="users.php" class="btn btn-secondary btn-sm">
                            <i class="fas fa-plus"></i> Add New
                        </a>
                    <?php endif; ?>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="id" value="<?php echo $edit_user ? $edit_user['id'] : 0; ?>">
                    
                    <div class="form-group">
                        <label for="username">Username *</label>
                        <input type="text" id="username" name="username" 
                               value="<?php echo $edit_user ? htmlspecialchars($edit_user['username']) : (isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''); ?>" 
                               class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="full_name">Full Name *</label>
                        <input type="text" id="full_name" name="full_name" 
                               value="<?php echo $edit_user ? htmlspecialchars($edit_user['full_name']) : (isset($_POST['full_name']) ? htmlspecialchars($_POST['full_name']) : ''); ?>" 
                               class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="email">Email *</label>
                        <input type="email" id="email" name="email" 
                               value="<?php echo $edit_user ? htmlspecialchars($edit_user['email']) : (isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''); ?>" 
                               class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="department">Department *</label>
                        <select id="department" name="department" class="form-control" required>
                            <option value="">Select Department</option>
                            <option value="admin" <?php echo ($edit_user && $edit_user['department'] == 'admin') || (isset($_POST['department']) && $_POST['department'] == 'admin') ? 'selected' : ''; ?>>Administrator</option>
                            <option value="lead" <?php echo ($edit_user && $edit_user['department'] == 'lead') || (isset($_POST['department']) && $_POST['department'] == 'lead') ? 'selected' : ''; ?>>Lead Department</option>
                            <option value="sales" <?php echo ($edit_user && $edit_user['department'] == 'sales') || (isset($_POST['department']) && $_POST['department'] == 'sales') ? 'selected' : ''; ?>>Sales Department</option>
                            <option value="delivery" <?php echo ($edit_user && $edit_user['department'] == 'delivery') || (isset($_POST['department']) && $_POST['department'] == 'delivery') ? 'selected' : ''; ?>>Delivery Department</option>
                            <option value="project" <?php echo ($edit_user && $edit_user['department'] == 'project') || (isset($_POST['department']) && $_POST['department'] == 'project') ? 'selected' : ''; ?>>Project Department</option>
                            <option value="upfollow" <?php echo ($edit_user && $edit_user['department'] == 'upfollow') || (isset($_POST['department']) && $_POST['department'] == 'upfollow') ? 'selected' : ''; ?>>Upscale Department</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="password">Password <?php echo $edit_user ? '(Leave blank to keep current)' : '*'; ?></label>
                        <input type="password" id="password" name="password" class="form-control" <?php echo $edit_user ? '' : 'required'; ?>>
                        <?php if ($edit_user): ?>
                            <div class="password-note">Leave password field empty to keep current password</div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="btn-group">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> <?php echo $edit_user ? 'Update User' : 'Add User'; ?>
                        </button>
                        <?php if ($edit_user): ?>
                            <a href="users.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Cancel
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Users List -->
            <div class="list-section">
                <h2><i class="fas fa-list"></i> All Users</h2>
                
                <div class="search-box">
                    <input type="text" id="userSearch" placeholder="Search users...">
                </div>
                
                <div class="table-container" style="overflow-x: auto;">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Username</th>
                                <th>Full Name</th>
                                <th>Email</th>
                                <th>Department</th>
                                <th>Created</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="usersTable">
                            <?php if ($users_result->num_rows == 0): ?>
                                <tr>
                                    <td colspan="7">
                                        <div class="empty-state">
                                            <i class="fas fa-users"></i>
                                            <h3>No Users Found</h3>
                                            <p>Add your first user using the form</p>
                                        </div>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php while($user = $users_result->fetch_assoc()): ?>
                                <tr>
                                    <td>#<?php echo str_pad($user['id'], 3, '0', STR_PAD_LEFT); ?></td>
                                    <td><strong><?php echo htmlspecialchars($user['username']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                                    <td>
                                        <?php 
                                        $badge_class = 'badge-' . $user['department'];
                                        echo '<span class="department-badge ' . $badge_class . '">' . ucfirst($user['department']) . '</span>';
                                        ?>
                                    </td>
                                    <td><?php echo date('M d, Y', strtotime($user['created_at'])); ?></td>
                                    <td>
                                        <div class="user-actions">
                                            <a href="?edit=<?php echo $user['id']; ?>" class="btn-icon btn-edit" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <?php if ($user['id'] != 1): ?>
                                            <a href="?delete=<?php echo $user['id']; ?>" class="btn-icon btn-delete" 
                                               title="Delete" onclick="return confirm('Are you sure you want to delete this user?')">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Search functionality
        document.getElementById('userSearch').addEventListener('keyup', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#usersTable tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        });
        
        // Auto-focus on username field when adding new user
        <?php if (!$edit_user): ?>
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('username').focus();
        });
        <?php endif; ?>
        
        // Password strength indicator
        const passwordInput = document.getElementById('password');
        if (passwordInput) {
            passwordInput.addEventListener('input', function() {
                const password = this.value;
                let strength = 0;
                
                if (password.length >= 8) strength++;
                if (/[A-Z]/.test(password)) strength++;
                if (/[0-9]/.test(password)) strength++;
                if (/[^A-Za-z0-9]/.test(password)) strength++;
                
                // You can add visual feedback here if needed
            });
        }
    </script>
</body>
</html>