<?php
require_once 'config.php';
requireDepartment(['lead', 'admin']);

$current_user_id = $_SESSION['user_id'];
$is_admin = ($_SESSION['department'] === 'admin');

// Handle form submission for new client
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_client'])) {
    $name = $conn->real_escape_string($_POST['name']);
    $phone = $conn->real_escape_string($_POST['phone']);
    $package = $conn->real_escape_string($_POST['package']);
    $note = $conn->real_escape_string($_POST['note']);
    $country = $conn->real_escape_string($_POST['country']);
    
    $stmt = $conn->prepare("INSERT INTO clients (name, phone, package, note, country, assigned_to) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("sssssi", $name, $phone, $package, $note, $country, $_SESSION['user_id']);
    
    if ($stmt->execute()) {
        $client_id = $stmt->insert_id;
        
        // Log the action
        $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'system', 'lead', ?, 'New client created in Lead department')");
        $action_stmt->bind_param("ii", $client_id, $_SESSION['user_id']);
        $action_stmt->execute();
        
        header('Location: lead_dept.php?success=1');
        exit();
    }
    
    $stmt->close();
}

// Handle refer to sales
if (isset($_GET['refer_to_sales'])) {
    $client_id = intval($_GET['refer_to_sales']);
    
    // Check if user owns this lead or is admin
    $check_stmt = $conn->prepare("SELECT assigned_to FROM clients WHERE id = ? AND current_department = 'lead'");
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows === 1) {
        $client = $check_result->fetch_assoc();
        
        // Allow if user is admin OR user owns the lead
        if ($is_admin || $client['assigned_to'] == $current_user_id) {
            $stmt = $conn->prepare("UPDATE clients SET current_department = 'sales', updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("i", $client_id);
            $stmt->execute();
            
            // Log the action
            $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'lead', 'sales', ?, 'Referred to Sales department')");
            $action_stmt->bind_param("ii", $client_id, $_SESSION['user_id']);
            $action_stmt->execute();
            
            header('Location: lead_dept.php?success=2');
            exit();
        } else {
            header('Location: lead_dept.php?error=1&message=' . urlencode('You can only refer your own leads to sales department.'));
            exit();
        }
    } else {
        header('Location: lead_dept.php?error=1&message=' . urlencode('Client not found or not in lead department.'));
        exit();
    }
}

// Handle edit client
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_client'])) {
    $client_id = intval($_POST['client_id']);
    $name = $conn->real_escape_string($_POST['name']);
    $phone = $conn->real_escape_string($_POST['phone']);
    $package = $conn->real_escape_string($_POST['package']);
    $note = $conn->real_escape_string($_POST['note']);
    $country = $conn->real_escape_string($_POST['country']);
    
    // Check if user owns this lead or is admin
    $check_stmt = $conn->prepare("SELECT assigned_to FROM clients WHERE id = ? AND current_department = 'lead'");
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows === 1) {
        $client = $check_result->fetch_assoc();
        
        // Allow if user is admin OR user owns the lead
        if ($is_admin || $client['assigned_to'] == $current_user_id) {
            $stmt = $conn->prepare("UPDATE clients SET name = ?, phone = ?, package = ?, note = ?, country = ?, updated_at = NOW() WHERE id = ?");
            $stmt->bind_param("sssssi", $name, $phone, $package, $note, $country, $client_id);
            
            if ($stmt->execute()) {
                // Log the action
                $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'lead', 'lead', ?, 'Client information updated in Lead department')");
                $action_stmt->bind_param("ii", $client_id, $_SESSION['user_id']);
                $action_stmt->execute();
                
                header('Location: lead_dept.php?success=3');
                exit();
            }
            $stmt->close();
        } else {
            header('Location: lead_dept.php?error=1&message=' . urlencode('You can only edit your own leads.'));
            exit();
        }
    }
    $check_stmt->close();
}

// Handle delete client
if (isset($_GET['delete_client'])) {
    $client_id = intval($_GET['delete_client']);
    
    // Check if user owns this lead or is admin
    $check_stmt = $conn->prepare("SELECT assigned_to FROM clients WHERE id = ? AND current_department = 'lead'");
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows === 1) {
        $client = $check_result->fetch_assoc();
        
        // Allow if user is admin OR user owns the lead
        if ($is_admin || $client['assigned_to'] == $current_user_id) {
            // First, delete client history
            $conn->query("DELETE FROM client_history WHERE client_id = $client_id");
            
            // Then delete the client
            $stmt = $conn->prepare("DELETE FROM clients WHERE id = ?");
            $stmt->bind_param("i", $client_id);
            
            if ($stmt->execute()) {
                header('Location: lead_dept.php?success=4');
                exit();
            }
            $stmt->close();
        } else {
            header('Location: lead_dept.php?error=1&message=' . urlencode('You can only delete your own leads.'));
            exit();
        }
    } else {
        header('Location: lead_dept.php?error=1&message=' . urlencode('Client not found or not in lead department.'));
        exit();
    }
    $check_stmt->close();
}

// Get client details for edit (if requested)
$edit_client = null;
$can_edit = false;
if (isset($_GET['edit'])) {
    $client_id = intval($_GET['edit']);
    $stmt = $conn->prepare("SELECT * FROM clients WHERE id = ? AND current_department = 'lead'");
    $stmt->bind_param("i", $client_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $edit_client = $result->fetch_assoc();
    
    // Check if user can edit this client
    if ($edit_client) {
        $can_edit = $is_admin || ($edit_client['assigned_to'] == $current_user_id);
        if (!$can_edit) {
            header('Location: lead_dept.php?error=1&message=' . urlencode('You can only edit your own leads.'));
            exit();
        }
    }
    $stmt->close();
}

// Get clients in lead department
// Regular users only see their own clients, admins see all
if ($is_admin) {
    $clients = $conn->query("SELECT c.*, u.full_name as assigned_to_name FROM clients c LEFT JOIN users u ON c.assigned_to = u.id WHERE c.current_department = 'lead' ORDER BY c.created_at DESC");
} else {
    $stmt = $conn->prepare("SELECT * FROM clients WHERE current_department = 'lead' AND assigned_to = ? ORDER BY created_at DESC");
    $stmt->bind_param("i", $current_user_id);
    $stmt->execute();
    $clients = $stmt->get_result();
}

// Get statistics for current user
if ($is_admin) {
    $user_stats = $conn->query("SELECT 
        COUNT(*) as total_clients,
        SUM(CASE WHEN package = 'paid' THEN 1 ELSE 0 END) as paid_clients,
        SUM(CASE WHEN package = 'free' THEN 1 ELSE 0 END) as free_clients,
        COUNT(DISTINCT assigned_to) as active_users
        FROM clients WHERE current_department = 'lead'")->fetch_assoc();
} else {
    $user_stats = $conn->query("SELECT 
        COUNT(*) as total_clients,
        SUM(CASE WHEN package = 'paid' THEN 1 ELSE 0 END) as paid_clients,
        SUM(CASE WHEN package = 'free' THEN 1 ELSE 0 END) as free_clients
        FROM clients WHERE current_department = 'lead' AND assigned_to = $current_user_id")->fetch_assoc();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Lead Department | Client Management System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .alert {
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1.5rem;
            font-weight: 500;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .alert-info {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: white;
            border-radius: 10px;
            padding: 2rem;
            max-width: 600px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .modal-header h3 {
            margin: 0;
            color: #333;
        }
        
        .btn-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #999;
        }
        
        .view-modal-content {
            max-width: 800px;
        }
        
        .client-details-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .detail-item {
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 5px;
        }
        
        .detail-item strong {
            display: block;
            margin-bottom: 0.5rem;
            color: #555;
        }
        
        .detail-item p {
            margin: 0;
            color: #333;
            font-size: 1.1rem;
        }
        
        .history-item {
            padding: 1rem;
            border-bottom: 1px solid #eee;
        }
        
        .history-item:last-child {
            border-bottom: none;
        }
        
        .history-date {
            color: #666;
            font-size: 0.9rem;
        }
        
        .history-action {
            color: #333;
            margin: 0.5rem 0;
        }
        
        .history-by {
            color: #666;
            font-size: 0.9rem;
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 10px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .stat-card h3 {
            color: #666;
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .stat-card .count {
            font-size: 2rem;
            font-weight: bold;
            color: #667eea;
        }
        
        .assigned-badge {
            display: inline-block;
            padding: 0.25rem 0.5rem;
            background: #e3f2fd;
            color: #1565c0;
            border-radius: 3px;
            font-size: 0.85rem;
            margin-top: 0.25rem;
        }
        
        .assigned-to-me {
            background: #d4edda;
            color: #155724;
        }
        
        .assigned-to-others {
            background: #fff3cd;
            color: #856404;
        }
        
        .permission-note {
            font-size: 0.9rem;
            color: #666;
            font-style: italic;
            margin-top: 0.5rem;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="main-content">
        <h1><i class="fas fa-user-plus"></i> Lead Department</h1>
        
        <?php if (isset($_GET['success'])): ?>
        <div class="alert alert-success">
            <?php 
            if ($_GET['success'] == 1) echo "Client added successfully!";
            if ($_GET['success'] == 2) echo "Client referred to Sales department successfully!";
            if ($_GET['success'] == 3) echo "Client updated successfully!";
            if ($_GET['success'] == 4) echo "Client deleted successfully!";
            ?>
        </div>
        <?php endif; ?>
        
        <?php if (isset($_GET['error'])): ?>
        <div class="alert alert-error">
            <i class="fas fa-exclamation-circle"></i> 
            <?php echo htmlspecialchars($_GET['message']); ?>
        </div>
        <?php endif; ?>
        
        <!-- User Statistics -->
        <div class="stats-cards">
            <div class="stat-card">
                <h3>My Leads</h3>
                <div class="count"><?php echo $user_stats['total_clients'] ?? 0; ?></div>
            </div>
            <div class="stat-card">
                <h3>Paid Packages</h3>
                <div class="count"><?php echo $user_stats['paid_clients'] ?? 0; ?></div>
            </div>
            <div class="stat-card">
                <h3>Free Packages</h3>
                <div class="count"><?php echo $user_stats['free_clients'] ?? 0; ?></div>
            </div>
            <?php if ($is_admin): ?>
            <div class="stat-card">
                <h3>Active Users</h3>
                <div class="count"><?php echo $user_stats['active_users'] ?? 0; ?></div>
            </div>
            <?php endif; ?>
        </div>
        
        <div class="form-container">
            <h2><i class="fas fa-plus-circle"></i> <?php echo $edit_client ? 'Edit Client' : 'Add New Client'; ?></h2>
            <?php if ($edit_client && !$can_edit): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i> You don't have permission to edit this lead. You can only edit leads assigned to you.
                </div>
            <?php endif; ?>
            
            <form method="POST" action="">
                <?php if ($edit_client): ?>
                    <input type="hidden" name="client_id" value="<?php echo $edit_client['id']; ?>">
                    <input type="hidden" name="edit_client" value="1">
                <?php else: ?>
                    <input type="hidden" name="add_client" value="1">
                <?php endif; ?>
                
                <div class="form-group">
                    <label for="name">Client Name *</label>
                    <input type="text" id="name" name="name" class="form-control" 
                           value="<?php echo $edit_client ? htmlspecialchars($edit_client['name']) : ''; ?>" required
                           <?php echo ($edit_client && !$can_edit) ? 'readonly' : ''; ?>>
                </div>
                <div class="form-group">
                    <label for="phone">Phone Number *</label>
                    <input type="tel" id="phone" name="phone" class="form-control" 
                           value="<?php echo $edit_client ? htmlspecialchars($edit_client['phone']) : ''; ?>" required
                           <?php echo ($edit_client && !$can_edit) ? 'readonly' : ''; ?>>
                </div>
                <div class="form-group">
                    <label for="package">Package *</label>
                    <select id="package" name="package" class="form-control" required
                            <?php echo ($edit_client && !$can_edit) ? 'disabled' : ''; ?>>
                        <option value="">Select Package</option>
                        <option value="free" <?php echo ($edit_client && $edit_client['package'] == 'free') ? 'selected' : ''; ?>>Free</option>
                        <option value="paid" <?php echo ($edit_client && $edit_client['package'] == 'paid') ? 'selected' : ''; ?>>Paid</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="country">Country</label>
                    <input type="text" id="country" name="country" class="form-control" 
                           value="<?php echo $edit_client ? htmlspecialchars($edit_client['country']) : ''; ?>"
                           <?php echo ($edit_client && !$can_edit) ? 'readonly' : ''; ?>>
                </div>
                <div class="form-group">
                    <label for="note">Note</label>
                    <textarea id="note" name="note" class="form-control" rows="3"
                              <?php echo ($edit_client && !$can_edit) ? 'readonly' : ''; ?>><?php echo $edit_client ? htmlspecialchars($edit_client['note']) : ''; ?></textarea>
                </div>
                <div class="btn-group">
                    <?php if ($edit_client && $can_edit): ?>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Update Client
                        </button>
                        <a href="lead_dept.php" class="btn btn-secondary">
                            <i class="fas fa-times"></i> Cancel
                        </a>
                    <?php elseif (!$edit_client): ?>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Add Client
                        </button>
                    <?php endif; ?>
                </div>
                
                <?php if ($edit_client && !$can_edit): ?>
                    <div class="permission-note">
                        <i class="fas fa-info-circle"></i> You are viewing a lead assigned to another user. Contact an administrator if you need to make changes.
                    </div>
                <?php endif; ?>
            </form>
        </div>
        
        <div class="table-container" style="margin-top: 2rem;">
            <h2><i class="fas fa-users"></i> Leads in Department</h2>
            
            <?php if ($clients->num_rows == 0): ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> 
                    <?php if ($is_admin): ?>
                        No leads found in Lead department. Leads will appear here when users add them.
                    <?php else: ?>
                        You haven't added any leads yet. Add your first lead above.
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Phone</th>
                            <th>Package</th>
                            <th>Country</th>
                            <th>Assigned To</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while($client = $clients->fetch_assoc()): 
                            $is_owner = ($client['assigned_to'] == $current_user_id);
                            $can_modify = $is_admin || $is_owner;
                            $assigned_name = $client['assigned_to_name'] ?? 'User #' . $client['assigned_to'];
                        ?>
                        <tr>
                            <td>#<?php echo str_pad($client['id'], 5, '0', STR_PAD_LEFT); ?></td>
                            <td><?php echo htmlspecialchars($client['name']); ?></td>
                            <td><?php echo htmlspecialchars($client['phone']); ?></td>
                            <td>
                                <span class="status-badge <?php echo $client['package'] === 'paid' ? 'status-completed' : 'status-pending'; ?>">
                                    <?php echo ucfirst($client['package']); ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($client['country']); ?></td>
                            <td>
                                <span class="assigned-badge <?php echo $is_owner ? 'assigned-to-me' : 'assigned-to-others'; ?>">
                                    <?php echo htmlspecialchars($assigned_name); ?>
                                    <?php if ($is_owner): ?>
                                        <i class="fas fa-user-check"></i>
                                    <?php else: ?>
                                        <i class="fas fa-user"></i>
                                    <?php endif; ?>
                                </span>
                            </td>
                            <td><?php echo date('M d, Y', strtotime($client['created_at'])); ?></td>
                            <td>
                                <div class="btn-group">
                                    <button class="btn btn-primary btn-sm" onclick="viewClient(<?php echo $client['id']; ?>)">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    
                                    <?php if ($can_modify): ?>
                                        <a href="?edit=<?php echo $client['id']; ?>" class="btn btn-warning btn-sm">
                                            <i class="fas fa-edit"></i> Edit
                                        </a>
                                    <?php else: ?>
                                        <button class="btn btn-warning btn-sm" onclick="alert('You can only edit leads assigned to you.')" title="Edit (Restricted)">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                    <?php endif; ?>
                                    
                                    <?php if ($can_modify): ?>
                                        <a href="?refer_to_sales=<?php echo $client['id']; ?>" class="btn btn-success btn-sm" onclick="return confirm('Refer this client to Sales department?')">
                                            <i class="fas fa-share"></i> Refer
                                        </a>
                                    <?php else: ?>
                                        <button class="btn btn-success btn-sm" onclick="alert('You can only refer your own leads to sales.')" title="Refer (Restricted)">
                                            <i class="fas fa-share"></i>
                                        </button>
                                    <?php endif; ?>
                                    
                                    <?php if ($can_modify): ?>
                                        <button class="btn btn-danger btn-sm" onclick="confirmDelete(<?php echo $client['id']; ?>, '<?php echo htmlspecialchars(addslashes($client['name'])); ?>')">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    <?php else: ?>
                                        <button class="btn btn-danger btn-sm" onclick="alert('You can only delete your own leads.')" title="Delete (Restricted)">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

    <!-- View Client Modal -->
    <div class="modal" id="viewModal">
        <div class="modal-content view-modal-content">
            <div class="modal-header">
                <h3>Client Details</h3>
                <button class="btn-close" onclick="closeViewModal()">&times;</button>
            </div>
            <div id="clientDetailsContent"></div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal" id="deleteModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Confirm Delete</h3>
                <button class="btn-close" onclick="closeDeleteModal()">&times;</button>
            </div>
            <div id="deleteConfirmContent">
                <p>Are you sure you want to delete this client?</p>
                <p><strong id="clientNameToDelete"></strong></p>
                <p style="color: #dc3545; font-size: 0.9rem;">
                    <i class="fas fa-exclamation-triangle"></i> This action cannot be undone.
                </p>
                <div class="btn-group" style="margin-top: 1.5rem;">
                    <button class="btn btn-danger" onclick="deleteClient()">
                        <i class="fas fa-trash"></i> Delete
                    </button>
                    <button class="btn btn-secondary" onclick="closeDeleteModal()">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        let clientToDelete = null;
        
        function viewClient(clientId) {
            fetch(`api/get_client.php?id=${clientId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        alert(data.error);
                        return;
                    }
                    
                    const modal = document.getElementById('viewModal');
                    const content = document.getElementById('clientDetailsContent');
                    
                    // Format date
                    const createdDate = new Date(data.created_at);
                    const updatedDate = new Date(data.updated_at);
                    
                    // Get assigned user info
                    fetch(`api/get_user_info.php?id=${data.assigned_to}`)
                        .then(response => response.json())
                        .then(userInfo => {
                            const assignedUserName = userInfo.full_name || 'User #' + data.assigned_to;
                            const isOwner = data.assigned_to == <?php echo $current_user_id; ?>;
                            const isAdmin = <?php echo $is_admin ? 'true' : 'false'; ?>;
                            
                            content.innerHTML = `
                                <div class="client-details-grid">
                                    <div class="detail-item">
                                        <strong>Client ID</strong>
                                        <p>#${data.id.toString().padStart(5, '0')}</p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Assigned To</strong>
                                        <p>
                                            <span class="assigned-badge ${isOwner ? 'assigned-to-me' : 'assigned-to-others'}">
                                                ${assignedUserName}
                                                ${isOwner ? '<i class="fas fa-user-check"></i>' : '<i class="fas fa-user"></i>'}
                                            </span>
                                        </p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Name</strong>
                                        <p>${escapeHtml(data.name)}</p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Phone</strong>
                                        <p>${escapeHtml(data.phone)}</p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Package</strong>
                                        <p><span class="status-badge ${data.package === 'paid' ? 'status-completed' : 'status-pending'}">${data.package}</span></p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Country</strong>
                                        <p>${escapeHtml(data.country || 'N/A')}</p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Status</strong>
                                        <p><span class="status-badge status-${data.status.replace('_', '-')}">${data.status.replace('_', ' ')}</span></p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Payment Status</strong>
                                        <p><span class="status-badge ${data.payment_status === 'collected' ? 'status-completed' : 'status-pending'}">${data.payment_status}</span></p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Current Department</strong>
                                        <p>${data.current_department}</p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Created Date</strong>
                                        <p>${createdDate.toLocaleDateString()} ${createdDate.toLocaleTimeString()}</p>
                                    </div>
                                    <div class="detail-item">
                                        <strong>Last Updated</strong>
                                        <p>${updatedDate.toLocaleDateString()} ${updatedDate.toLocaleTimeString()}</p>
                                    </div>
                                </div>
                                
                                <div class="detail-item">
                                    <strong>Note</strong>
                                    <p>${escapeHtml(data.note || 'No notes available')}</p>
                                </div>
                                
                                <div class="sales-actions" style="background: #f8f9fa; padding: 1.5rem; border-radius: 10px; margin: 1.5rem 0;">
                                    <h4>Actions</h4>
                                    <div class="quick-actions" style="display: flex; gap: 1rem; flex-wrap: wrap; margin-top: 1rem;">
                                        ${(isAdmin || isOwner) ? `
                                        <a href="?edit=${data.id}" class="btn btn-warning">
                                            <i class="fas fa-edit"></i> Edit
                                        </a>
                                        <a href="?refer_to_sales=${data.id}" class="btn btn-success" onclick="return confirm('Refer this client to Sales department?')">
                                            <i class="fas fa-share"></i> Refer to Sales
                                        </a>
                                        <button class="btn btn-danger" onclick="confirmDelete(${data.id}, '${escapeHtml(data.name)}')">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                        ` : `
                                        <button class="btn btn-warning" onclick="alert('You can only edit leads assigned to you.')">
                                            <i class="fas fa-edit"></i> Edit (Restricted)
                                        </button>
                                        <button class="btn btn-success" onclick="alert('You can only refer your own leads to sales.')">
                                            <i class="fas fa-share"></i> Refer (Restricted)
                                        </button>
                                        <button class="btn btn-danger" onclick="alert('You can only delete your own leads.')">
                                            <i class="fas fa-trash"></i> Delete (Restricted)
                                        </button>
                                        `}
                                    </div>
                                    ${!(isAdmin || isOwner) ? `
                                    <div class="permission-note" style="margin-top: 1rem;">
                                        <i class="fas fa-info-circle"></i> This lead is assigned to another user. You can only view it.
                                    </div>
                                    ` : ''}
                                </div>
                                
                                <h4 style="margin-top: 2rem; margin-bottom: 1rem;">Client History</h4>
                                <div id="clientHistory"></div>
                            `;
                            
                            // Load client history
                            fetch(`api/get_client_history.php?id=${clientId}`)
                                .then(response => response.json())
                                .then(history => {
                                    const historyDiv = document.getElementById('clientHistory');
                                    if (history.length === 0) {
                                        historyDiv.innerHTML = '<p>No history available for this client.</p>';
                                        return;
                                    }
                                    
                                    let historyHTML = '';
                                    history.forEach(record => {
                                        const actionDate = new Date(record.action_date);
                                        historyHTML += `
                                            <div class="history-item">
                                                <div class="history-date">${actionDate.toLocaleDateString()} ${actionDate.toLocaleTimeString()}</div>
                                                <div class="history-action">${escapeHtml(record.notes)}</div>
                                                <div class="history-by">Action by: ${escapeHtml(record.action_by_name)}</div>
                                            </div>
                                        `;
                                    });
                                    historyDiv.innerHTML = historyHTML;
                                })
                                .catch(error => {
                                    console.error('Error loading history:', error);
                                    document.getElementById('clientHistory').innerHTML = '<p>Unable to load history.</p>';
                                });
                            
                            modal.classList.add('active');
                        })
                        .catch(error => {
                            console.error('Error loading user info:', error);
                            content.innerHTML = '<p>Error loading client details. Please try again.</p>';
                        });
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Failed to load client details');
                });
        }
        
        function closeViewModal() {
            document.getElementById('viewModal').classList.remove('active');
        }
        
        function confirmDelete(clientId, clientName) {
            // Check if user has permission
            const isAdmin = <?php echo $is_admin ? 'true' : 'false'; ?>;
            const userId = <?php echo $current_user_id; ?>;
            
            if (!isAdmin) {
                // Check if this client belongs to current user
                fetch(`api/get_client.php?id=${clientId}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.assigned_to == userId) {
                            showDeleteModal(clientId, clientName);
                        } else {
                            alert('You can only delete your own leads.');
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        alert('Failed to verify permissions.');
                    });
            } else {
                showDeleteModal(clientId, clientName);
            }
        }
        
        function showDeleteModal(clientId, clientName) {
            clientToDelete = clientId;
            document.getElementById('clientNameToDelete').textContent = clientName;
            document.getElementById('deleteModal').classList.add('active');
        }
        
        function closeDeleteModal() {
            document.getElementById('deleteModal').classList.remove('active');
            clientToDelete = null;
        }
        
        function deleteClient() {
            if (clientToDelete) {
                window.location.href = `?delete_client=${clientToDelete}`;
            }
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const viewModal = document.getElementById('viewModal');
            const deleteModal = document.getElementById('deleteModal');
            
            if (event.target === viewModal) {
                closeViewModal();
            }
            if (event.target === deleteModal) {
                closeDeleteModal();
            }
        }
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(event) {
            const viewModal = document.getElementById('viewModal');
            const deleteModal = document.getElementById('deleteModal');
            
            if (event.key === 'Escape') {
                if (viewModal.classList.contains('active')) {
                    closeViewModal();
                }
                if (deleteModal.classList.contains('active')) {
                    closeDeleteModal();
                }
            }
        });
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>