<?php
require_once 'config.php';
requireDepartment(['sales', 'admin']);

// Handle refer to delivery
if (isset($_GET['refer_to_delivery'])) {
    $client_id = intval($_GET['refer_to_delivery']);
    
    $stmt = $conn->prepare("UPDATE clients SET current_department = 'delivery', updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("i", $client_id);
    $stmt->execute();
    
    // Log the action
    $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'sales', 'delivery', ?, 'Referred to Delivery department')");
    $action_stmt->bind_param("ii", $client_id, $_SESSION['user_id']);
    $action_stmt->execute();
    
    header('Location: sales_dept.php?success=1');
    exit();
}

// Handle send back to lead
if (isset($_GET['send_back_lead'])) {
    $client_id = intval($_GET['send_back_lead']);
    
    $stmt = $conn->prepare("UPDATE clients SET current_department = 'lead', updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("i", $client_id);
    $stmt->execute();
    
    // Log the action
    $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'sales', 'lead', ?, 'Sent back to Lead department')");
    $action_stmt->bind_param("ii", $client_id, $_SESSION['user_id']);
    $action_stmt->execute();
    
    header('Location: sales_dept.php?success=2');
    exit();
}

// Handle add sales note
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_sales_note'])) {
    $client_id = intval($_POST['client_id']);
    $sales_note = $conn->real_escape_string($_POST['sales_note']);
    
    // Update client note
    $stmt = $conn->prepare("UPDATE clients SET note = CONCAT(COALESCE(note, ''), '\nSales Update: ', ?), updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("si", $sales_note, $client_id);
    $stmt->execute();
    
    // Log the action
    $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'sales', 'sales', ?, 'Sales note added: ' || ?)");
    $action_stmt->bind_param("iis", $client_id, $_SESSION['user_id'], $sales_note);
    $action_stmt->execute();
    
    header('Location: sales_dept.php?success=3&client_id=' . $client_id);
    exit();
}

// Handle update package type
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_package'])) {
    $client_id = intval($_POST['client_id']);
    $package = $conn->real_escape_string($_POST['package']);
    
    $stmt = $conn->prepare("UPDATE clients SET package = ?, updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("si", $package, $client_id);
    $stmt->execute();
    
    // Log the action
    $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'sales', 'sales', ?, 'Package updated to ' || ?)");
    $action_stmt->bind_param("iis", $client_id, $_SESSION['user_id'], $package);
    $action_stmt->execute();
    
    header('Location: sales_dept.php?success=4&client_id=' . $client_id);
    exit();
}

// Handle add package price/deal
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_deal'])) {
    $client_id = intval($_POST['client_id']);
    $package_price = floatval($_POST['package_price']);
    $deal_status = $conn->real_escape_string($_POST['deal_status']);
    $discount_percentage = isset($_POST['discount_percentage']) ? floatval($_POST['discount_percentage']) : 0;
    $deal_notes = $conn->real_escape_string($_POST['deal_notes']);
    $payment_method = $conn->real_escape_string($_POST['payment_method']);
    
    // Calculate final amount
    $discount_amount = $package_price * ($discount_percentage / 100);
    $final_amount = $package_price - $discount_amount;
    
    // If deal is accepted/completed, set payment status to collected
    $payment_status = ($deal_status == 'accepted' || $deal_status == 'completed') ? 'collected' : 'pending';
    
    // Update client with deal information
    $stmt = $conn->prepare("UPDATE clients SET 
        package_price = ?, 
        deal_status = ?, 
        discount_percentage = ?, 
        final_amount = ?,
        payment_method = ?,
        deal_notes = CONCAT(COALESCE(deal_notes, ''), '\n', ?),
        payment_status = ?,
        updated_at = NOW(),
        deal_closed_at = CASE WHEN ? IN ('accepted', 'completed') THEN NOW() ELSE NULL END
        WHERE id = ?");
    
    $stmt->bind_param("ssddssssi", 
        $package_price, 
        $deal_status, 
        $discount_percentage, 
        $final_amount,
        $payment_method,
        $deal_notes,
        $payment_status,
        $deal_status,
        $client_id
    );
    
    if ($stmt->execute()) {
        // Log the action
        $log_notes = "Deal updated: Package Price: $package_price, Status: " . ucfirst($deal_status);
        if ($discount_percentage > 0) {
            $log_notes .= ", Discount: $discount_percentage%, Final Amount: $final_amount";
        }
        
        $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'sales', 'sales', ?, ?)");
        $action_stmt->bind_param("iis", $client_id, $_SESSION['user_id'], $log_notes);
        $action_stmt->execute();
        
        header('Location: sales_dept.php?success=5&client_id=' . $client_id);
        exit();
    }
    
    $stmt->close();
}

// Handle close deal
if (isset($_GET['close_deal'])) {
    $client_id = intval($_GET['close_deal']);
    
    $stmt = $conn->prepare("UPDATE clients SET 
        deal_status = 'completed',
        payment_status = 'collected',
        deal_closed_at = NOW(),
        updated_at = NOW()
        WHERE id = ?");
    
    $stmt->bind_param("i", $client_id);
    
    if ($stmt->execute()) {
        // Log the action
        $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'sales', 'sales', ?, 'Deal marked as completed and payment collected')");
        $action_stmt->bind_param("ii", $client_id, $_SESSION['user_id']);
        $action_stmt->execute();
        
        header('Location: sales_dept.php?success=6&client_id=' . $client_id);
        exit();
    }
    
    $stmt->close();
}

// Get clients in sales department
$clients = $conn->query("SELECT * FROM clients WHERE current_department = 'sales' ORDER BY created_at DESC");

// Calculate sales statistics including revenue
$revenue_stats = $conn->query("
    SELECT 
        COUNT(*) as total_clients,
        SUM(CASE WHEN package = 'paid' THEN 1 ELSE 0 END) as paid_clients,
        SUM(CASE WHEN package = 'free' THEN 1 ELSE 0 END) as free_clients,
        SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today_clients,
        SUM(final_amount) as total_revenue,
        AVG(final_amount) as avg_deal_size,
        SUM(CASE WHEN deal_status = 'completed' THEN 1 ELSE 0 END) as completed_deals,
        SUM(CASE WHEN deal_status = 'accepted' THEN 1 ELSE 0 END) as accepted_deals
    FROM clients 
    WHERE current_department = 'sales'
")->fetch_assoc();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sales Department | Client Management System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .alert {
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1.5rem;
            font-weight: 500;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-info {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: white;
            border-radius: 10px;
            padding: 2rem;
            max-width: 800px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .modal-header h3 {
            margin: 0;
            color: #333;
        }
        
        .btn-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #999;
        }
        
        .client-details-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .detail-item {
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 5px;
        }
        
        .detail-item strong {
            display: block;
            margin-bottom: 0.5rem;
            color: #555;
        }
        
        .detail-item p {
            margin: 0;
            color: #333;
            font-size: 1.1rem;
        }
        
        .history-item {
            padding: 1rem;
            border-bottom: 1px solid #eee;
        }
        
        .history-item:last-child {
            border-bottom: none;
        }
        
        .history-date {
            color: #666;
            font-size: 0.9rem;
        }
        
        .history-action {
            color: #333;
            margin: 0.5rem 0;
        }
        
        .history-by {
            color: #666;
            font-size: 0.9rem;
        }
        
        .sales-actions {
            background: #f8f9fa;
            padding: 1.5rem;
            border-radius: 10px;
            margin: 1.5rem 0;
        }
        
        .sales-actions h4 {
            margin-bottom: 1rem;
            color: #333;
        }
        
        .quick-actions {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
            margin-top: 1rem;
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 10px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            text-align: center;
        }
        
        .stat-card h3 {
            color: #666;
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .stat-card .count {
            font-size: 2rem;
            font-weight: bold;
            color: #667eea;
        }
        
        .stat-card .revenue {
            font-size: 1.5rem;
            font-weight: bold;
            color: #28a745;
        }
        
        .form-inline {
            display: flex;
            gap: 1rem;
            align-items: flex-end;
        }
        
        .form-inline .form-group {
            flex: 1;
            margin-bottom: 0;
        }
        
        .form-inline label {
            display: block;
            margin-bottom: 0.5rem;
            color: #555;
            font-weight: 500;
        }
        
        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.9rem;
        }
        
        .deal-status {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
        }
        
        .deal-pending { background: #fff3cd; color: #856404; }
        .deal-negotiating { background: #d1ecf1; color: #0c5460; }
        .deal-accepted { background: #d4edda; color: #155724; }
        .deal-rejected { background: #f8d7da; color: #721c24; }
        .deal-completed { background: #cce5ff; color: #004085; }
        
        .price-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.75rem;
            background: #e8f5e9;
            color: #2e7d32;
            border-radius: 20px;
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .price-badge.free {
            background: #fff3e0;
            color: #ef6c00;
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }
        
        .calculation-box {
            background: #f8f9fa;
            padding: 1rem;
            border-radius: 5px;
            margin: 1rem 0;
        }
        
        .calculation-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 1px solid #dee2e6;
        }
        
        .calculation-row:last-child {
            border-bottom: none;
            font-weight: bold;
            font-size: 1.1rem;
            color: #333;
        }
        
        .discount-badge {
            display: inline-block;
            padding: 0.1rem 0.5rem;
            background: #dc3545;
            color: white;
            border-radius: 3px;
            font-size: 0.8rem;
            font-weight: 500;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="main-content">
        <h1><i class="fas fa-chart-line"></i> Sales Department</h1>
        
        <?php 
        if (isset($_GET['success'])):
            $messages = [
                1 => "Client referred to Delivery department successfully!",
                2 => "Client sent back to Lead department successfully!",
                3 => "Sales note added successfully!",
                4 => "Package updated successfully!",
                5 => "Deal information saved successfully!",
                6 => "Deal marked as completed successfully!"
            ];
        ?>
        <div class="alert alert-success">
            <?php echo $messages[$_GET['success']]; ?>
        </div>
        <?php endif; ?>
        
        <!-- Sales Statistics with Revenue -->
        <div class="stats-cards">
            <div class="stat-card">
                <h3>Total Clients</h3>
                <div class="count"><?php echo $revenue_stats['total_clients']; ?></div>
            </div>
            <div class="stat-card">
                <h3>Paid Packages</h3>
                <div class="count"><?php echo $revenue_stats['paid_clients']; ?></div>
            </div>
            <div class="stat-card">
                <h3>Total Revenue</h3>
                <div class="revenue">PKR<?php echo number_format($revenue_stats['total_revenue'] ?? 0, 2); ?></div>
            </div>
            <div class="stat-card">
                <h3>Avg Deal Size</h3>
                <div class="revenue">PKR<?php echo number_format($revenue_stats['avg_deal_size'] ?? 0, 2); ?></div>
            </div>
            <div class="stat-card">
                <h3>Completed Deals</h3>
                <div class="count"><?php echo $revenue_stats['completed_deals']; ?></div>
            </div>
            <div class="stat-card">
                <h3>Today's Clients</h3>
                <div class="count"><?php echo $revenue_stats['today_clients']; ?></div>
            </div>
        </div>
        
        <div class="table-container">
            <h2><i class="fas fa-users"></i> Clients in Sales Department</h2>
            
            <?php if ($clients->num_rows == 0): ?>
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> No clients found in Sales department. Clients will appear here when referred from Lead department.
                </div>
            <?php else: ?>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Package</th>
                            <th>Price</th>
                            <th>Deal Status</th>
                            <th>Final Amount</th>
                            <th>Payment</th>
                            <th>Days in Sales</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while($client = $clients->fetch_assoc()): 
                            // Calculate days in sales
                            $days_in_sales = floor((time() - strtotime($client['updated_at'])) / (60 * 60 * 24));
                        ?>
                        <tr>
                            <td>#<?php echo str_pad($client['id'], 5, '0', STR_PAD_LEFT); ?></td>
                            <td><strong><?php echo htmlspecialchars($client['name']); ?></strong></td>
                            <td>
                                <span class="status-badge <?php echo $client['package'] === 'paid' ? 'status-completed' : 'status-pending'; ?>">
                                    <?php echo ucfirst($client['package']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($client['package'] === 'paid' && $client['package_price']): ?>
                                    <span class="price-badge">
                                        <i class="fas fa-dollar-sign"></i>
                                        <?php echo number_format($client['package_price'], 2); ?>
                                        <?php if ($client['discount_percentage'] > 0): ?>
                                            <span class="discount-badge">-<?php echo $client['discount_percentage']; ?>%</span>
                                        <?php endif; ?>
                                    </span>
                                <?php elseif ($client['package'] === 'paid'): ?>
                                    <span class="price-badge" style="background: #ffebee; color: #c62828;">
                                        <i class="fas fa-exclamation-circle"></i> Set Price
                                    </span>
                                <?php else: ?>
                                    <span class="price-badge free">
                                        <i class="fas fa-gift"></i> Free
                                    </span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($client['deal_status']): ?>
                                    <span class="deal-status deal-<?php echo $client['deal_status']; ?>">
                                        <?php echo ucfirst($client['deal_status']); ?>
                                    </span>
                                <?php else: ?>
                                    <span class="deal-status deal-pending">Pending</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($client['final_amount']): ?>
                                    <strong>PKR<?php echo number_format($client['final_amount'], 2); ?></strong>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="status-badge <?php echo $client['payment_status'] === 'collected' ? 'status-completed' : 'status-pending'; ?>">
                                    <?php echo ucfirst($client['payment_status']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge <?php echo $days_in_sales > 7 ? 'status-pending' : 'status-in-progress'; ?>">
                                    <?php echo $days_in_sales; ?> days
                                </span>
                            </td>
                            <td>
                                <div class="btn-group">
                                    <button class="btn btn-primary btn-sm" onclick="viewClient(<?php echo $client['id']; ?>)">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    <button class="btn btn-info btn-sm" onclick="addSalesNote(<?php echo $client['id']; ?>)">
                                        <i class="fas fa-comment"></i> Note
                                    </button>
                                    <?php if ($client['package'] === 'paid'): ?>
                                        <button class="btn btn-warning btn-sm" onclick="addPackagePrice(<?php echo $client['id']; ?>, '<?php echo $client['deal_status'] ?: 'pending'; ?>')">
                                            <i class="fas fa-dollar-sign"></i> Deal
                                        </button>
                                    <?php else: ?>
                                        <button class="btn btn-warning btn-sm" onclick="updatePackage(<?php echo $client['id']; ?>, '<?php echo $client['package']; ?>')">
                                            <i class="fas fa-tag"></i> Package
                                        </button>
                                    <?php endif; ?>
                                    <a href="?send_back_lead=<?php echo $client['id']; ?>" class="btn btn-secondary btn-sm" onclick="return confirm('Send this client back to Lead department?')">
                                        <i class="fas fa-arrow-left"></i>
                                    </a>
                                    <?php if ($client['deal_status'] === 'accepted' || $client['deal_status'] === 'completed'): ?>
                                        <a href="?refer_to_delivery=<?php echo $client['id']; ?>" class="btn btn-success btn-sm" onclick="return confirm('Refer this client to Delivery department?')">
                                            <i class="fas fa-arrow-right"></i>
                                        </a>
                                    <?php else: ?>
                                        <button class="btn btn-success btn-sm" onclick="alert('Please accept the deal first before referring to delivery.')">
                                            <i class="fas fa-arrow-right"></i>
                                        </button>
                                    <?php endif; ?>
                                    <?php if ($client['deal_status'] === 'accepted' && $client['payment_status'] !== 'collected'): ?>
                                        <a href="?close_deal=<?php echo $client['id']; ?>" class="btn btn-danger btn-sm" onclick="return confirm('Mark this deal as completed and payment collected?')">
                                            <i class="fas fa-check-circle"></i> Close
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

    <!-- View Client Modal -->
    <div class="modal" id="viewModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Client Details</h3>
                <button class="btn-close" onclick="closeViewModal()">&times;</button>
            </div>
            <div id="clientDetailsContent"></div>
        </div>
    </div>

    <!-- Add Sales Note Modal -->
    <div class="modal" id="noteModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add Sales Note</h3>
                <button class="btn-close" onclick="closeNoteModal()">&times;</button>
            </div>
            <form id="noteForm" method="POST" action="">
                <input type="hidden" name="client_id" id="noteClientId">
                <input type="hidden" name="add_sales_note" value="1">
                <div class="form-group">
                    <label for="sales_note">Sales Note *</label>
                    <textarea id="sales_note" name="sales_note" class="form-control" rows="4" required placeholder="Enter sales notes, follow-up details, or client feedback..."></textarea>
                </div>
                <div class="btn-group" style="margin-top: 1.5rem;">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Save Note
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="closeNoteModal()">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Update Package Modal -->
    <div class="modal" id="packageModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Update Package Type</h3>
                <button class="btn-close" onclick="closePackageModal()">&times;</button>
            </div>
            <form id="packageForm" method="POST" action="">
                <input type="hidden" name="client_id" id="packageClientId">
                <input type="hidden" name="update_package" value="1">
                <div class="form-group">
                    <label for="package">Select Package Type *</label>
                    <select id="package" name="package" class="form-control" required>
                        <option value="">Select Package</option>
                        <option value="free">Free Package</option>
                        <option value="paid">Paid Package</option>
                    </select>
                </div>
                <div class="btn-group" style="margin-top: 1.5rem;">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Update Package
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="closePackageModal()">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Add Package Price/Deal Modal -->
    <div class="modal" id="priceModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Package Deal Management</h3>
                <button class="btn-close" onclick="closePriceModal()">&times;</button>
            </div>
            <form id="priceForm" method="POST" action="">
                <input type="hidden" name="client_id" id="priceClientId">
                <input type="hidden" name="add_deal" value="1">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="package_price">Package Price (PKR) *</label>
                        <input type="number" id="package_price" name="package_price" class="form-control" step="0.01" min="0" required placeholder="0.00">
                    </div>
                    
                    <div class="form-group">
                        <label for="discount_percentage">Discount (%)</label>
                        <input type="number" id="discount_percentage" name="discount_percentage" class="form-control" step="0.01" min="0" max="100" value="0" placeholder="0">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="deal_status">Deal Status *</label>
                        <select id="deal_status" name="deal_status" class="form-control" required>
                            <option value="pending">Pending</option>
                            <option value="negotiating">Negotiating</option>
                            <option value="accepted">Accepted</option>
                            <option value="rejected">Rejected</option>
                            <option value="completed">Completed</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="payment_method">Payment Method</label>
                        <select id="payment_method" name="payment_method" class="form-control">
                            <option value="">Select Method</option>
                            <option value="cash">Cash</option>
                            <option value="credit_card">Credit Card</option>
                            <option value="bank_transfer">Bank Transfer</option>
                            <option value="check">Check</option>
                            <option value="online">Online Payment</option>
                        </select>
                    </div>
                </div>
                
                <div class="calculation-box">
                    <div class="calculation-row">
                        <span>Package Price:</span>
                        <span id="calcPackagePrice">PKR 0.00</span>
                    </div>
                    <div class="calculation-row">
                        <span>Discount (<span id="calcDiscountPercent">0</span>%):</span>
                        <span id="calcDiscountAmount">-PKR 0.00</span>
                    </div>
                    <div class="calculation-row">
                        <span>Final Amount:</span>
                        <span id="calcFinalAmount">PKR 0.00</span>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="deal_notes">Deal Notes</label>
                    <textarea id="deal_notes" name="deal_notes" class="form-control" rows="3" placeholder="Add notes about the deal, negotiations, or special terms..."></textarea>
                </div>
                
                <div class="btn-group" style="margin-top: 1.5rem;">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Save Deal
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="closePriceModal()">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        function viewClient(clientId) {
            fetch(`api/get_client.php?id=${clientId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        alert(data.error);
                        return;
                    }
                    
                    const modal = document.getElementById('viewModal');
                    const content = document.getElementById('clientDetailsContent');
                    
                    // Format dates
                    const createdDate = new Date(data.created_at);
                    const updatedDate = new Date(data.updated_at);
                    const dealClosedDate = data.deal_closed_at ? new Date(data.deal_closed_at) : null;
                    
                    let dealInfo = '';
                    if (data.package === 'paid') {
                        dealInfo = `
                            <div class="detail-item">
                                <strong>Package Price</strong>
                                <p>$${data.package_price ? parseFloat(data.package_price).toFixed(2) : 'Not set'}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Deal Status</strong>
                                <p><span class="deal-status deal-${data.deal_status || 'pending'}">${data.deal_status ? data.deal_status.charAt(0).toUpperCase() + data.deal_status.slice(1) : 'Pending'}</span></p>
                            </div>
                            <div class="detail-item">
                                <strong>Discount</strong>
                                <p>${data.discount_percentage ? data.discount_percentage + '%' : '0%'}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Final Amount</strong>
                                <p>$${data.final_amount ? parseFloat(data.final_amount).toFixed(2) : '0.00'}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Payment Method</strong>
                                <p>${data.payment_method ? data.payment_method.charAt(0).toUpperCase() + data.payment_method.slice(1) : 'Not set'}</p>
                            </div>
                        `;
                        
                        if (data.deal_closed_at) {
                            dealInfo += `
                                <div class="detail-item">
                                    <strong>Deal Closed Date</strong>
                                    <p>${dealClosedDate.toLocaleDateString()} ${dealClosedDate.toLocaleTimeString()}</p>
                                </div>
                            `;
                        }
                    }
                    
                    content.innerHTML = `
                        <div class="client-details-grid">
                            <div class="detail-item">
                                <strong>Client ID</strong>
                                <p>#${data.id.toString().padStart(5, '0')}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Name</strong>
                                <p>${escapeHtml(data.name)}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Phone</strong>
                                <p>${escapeHtml(data.phone)}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Package</strong>
                                <p><span class="status-badge ${data.package === 'paid' ? 'status-completed' : 'status-pending'}">${data.package}</span></p>
                            </div>
                            <div class="detail-item">
                                <strong>Country</strong>
                                <p>${escapeHtml(data.country || 'N/A')}</p>
                            </div>
                            ${dealInfo}
                            <div class="detail-item">
                                <strong>Status</strong>
                                <p><span class="status-badge status-${data.status.replace('_', '-')}">${data.status.replace('_', ' ')}</span></p>
                            </div>
                            <div class="detail-item">
                                <strong>Payment Status</strong>
                                <p><span class="status-badge ${data.payment_status === 'collected' ? 'status-completed' : 'status-pending'}">${data.payment_status}</span></p>
                            </div>
                            <div class="detail-item">
                                <strong>Current Department</strong>
                                <p>${data.current_department}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Created Date</strong>
                                <p>${createdDate.toLocaleDateString()} ${createdDate.toLocaleTimeString()}</p>
                            </div>
                            <div class="detail-item">
                                <strong>Last Updated</strong>
                                <p>${updatedDate.toLocaleDateString()} ${updatedDate.toLocaleTimeString()}</p>
                            </div>
                        </div>
                        
                        ${data.deal_notes ? `
                        <div class="detail-item">
                            <strong>Deal Notes</strong>
                            <p style="white-space: pre-wrap;">${escapeHtml(data.deal_notes)}</p>
                        </div>
                        ` : ''}
                        
                        <div class="detail-item">
                            <strong>Full Notes</strong>
                            <p style="white-space: pre-wrap;">${escapeHtml(data.note || 'No notes available')}</p>
                        </div>
                        
                        <div class="sales-actions">
                            <h4>Quick Actions</h4>
                            <div class="quick-actions">
                                <a href="?send_back_lead=${data.id}" class="btn btn-secondary" onclick="return confirm('Send this client back to Lead department?')">
                                    <i class="fas fa-arrow-left"></i> Back to Lead
                                </a>
                                ${data.package === 'paid' ? `
                                <button class="btn btn-warning" onclick="addPackagePrice(${data.id}, '${data.deal_status || 'pending'}')">
                                    <i class="fas fa-dollar-sign"></i> Manage Deal
                                </button>
                                ` : `
                                <button class="btn btn-warning" onclick="updatePackage(${data.id}, '${data.package}')">
                                    <i class="fas fa-tag"></i> Update Package
                                </button>
                                `}
                                ${(data.deal_status === 'accepted' || data.deal_status === 'completed') ? `
                                <a href="?refer_to_delivery=${data.id}" class="btn btn-success" onclick="return confirm('Refer this client to Delivery department?')">
                                    <i class="fas fa-arrow-right"></i> Refer to Delivery
                                </a>
                                ` : `
                                <button class="btn btn-success" onclick="alert('Please accept the deal first before referring to delivery.')">
                                    <i class="fas fa-arrow-right"></i> Refer to Delivery
                                </button>
                                `}
                                <button class="btn btn-info" onclick="addSalesNote(${data.id})">
                                    <i class="fas fa-comment"></i> Add Note
                                </button>
                                ${data.deal_status === 'accepted' && data.payment_status !== 'collected' ? `
                                <a href="?close_deal=${data.id}" class="btn btn-danger" onclick="return confirm('Mark this deal as completed and payment collected?')">
                                    <i class="fas fa-check-circle"></i> Close Deal
                                </a>
                                ` : ''}
                            </div>
                        </div>
                        
                        <h4 style="margin-top: 2rem; margin-bottom: 1rem;">Client History</h4>
                        <div id="clientHistory"></div>
                    `;
                    
                    // Load client history
                    fetch(`api/get_client_history.php?id=${clientId}`)
                        .then(response => response.json())
                        .then(history => {
                            const historyDiv = document.getElementById('clientHistory');
                            if (history.length === 0) {
                                historyDiv.innerHTML = '<p>No history available for this client.</p>';
                                return;
                            }
                            
                            let historyHTML = '';
                            history.forEach(record => {
                                const actionDate = new Date(record.action_date);
                                historyHTML += `
                                    <div class="history-item">
                                        <div class="history-date">${actionDate.toLocaleDateString()} ${actionDate.toLocaleTimeString()}</div>
                                        <div class="history-action">${escapeHtml(record.notes)}</div>
                                        <div class="history-by">Action by: ${escapeHtml(record.action_by_name)}</div>
                                    </div>
                                `;
                            });
                            historyDiv.innerHTML = historyHTML;
                        })
                        .catch(error => {
                            console.error('Error loading history:', error);
                            document.getElementById('clientHistory').innerHTML = '<p>Unable to load history.</p>';
                        });
                    
                    modal.classList.add('active');
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Failed to load client details');
                });
        }
        
        function closeViewModal() {
            document.getElementById('viewModal').classList.remove('active');
        }
        
        function addSalesNote(clientId) {
            document.getElementById('noteClientId').value = clientId;
            document.getElementById('sales_note').value = '';
            document.getElementById('noteModal').classList.add('active');
            document.getElementById('sales_note').focus();
        }
        
        function closeNoteModal() {
            document.getElementById('noteModal').classList.remove('active');
        }
        
        function updatePackage(clientId, currentPackage) {
            document.getElementById('packageClientId').value = clientId;
            document.getElementById('package').value = currentPackage;
            document.getElementById('packageModal').classList.add('active');
        }
        
        function closePackageModal() {
            document.getElementById('packageModal').classList.remove('active');
        }
        
        function addPackagePrice(clientId, currentStatus) {
            // Fetch client data to pre-fill the form
            fetch(`api/get_client.php?id=${clientId}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('priceClientId').value = clientId;
                    
                    if (data.package_price) {
                        document.getElementById('package_price').value = data.package_price;
                    }
                    if (data.discount_percentage) {
                        document.getElementById('discount_percentage').value = data.discount_percentage;
                    }
                    if (data.deal_status) {
                        document.getElementById('deal_status').value = data.deal_status;
                    } else {
                        document.getElementById('deal_status').value = currentStatus;
                    }
                    if (data.payment_method) {
                        document.getElementById('payment_method').value = data.payment_method;
                    }
                    if (data.deal_notes) {
                        document.getElementById('deal_notes').value = data.deal_notes;
                    }
                    
                    // Update calculations
                    calculateDealAmount();
                    
                    document.getElementById('priceModal').classList.add('active');
                    document.getElementById('package_price').focus();
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Failed to load client data');
                });
        }
        
        function closePriceModal() {
            document.getElementById('priceModal').classList.remove('active');
        }
        
        function calculateDealAmount() {
            const price = parseFloat(document.getElementById('package_price').value) || 0;
            const discount = parseFloat(document.getElementById('discount_percentage').value) || 0;
            
            const discountAmount = price * (discount / 100);
            const finalAmount = price - discountAmount;
            
            document.getElementById('calcPackagePrice').textContent = '$' + price.toFixed(2);
            document.getElementById('calcDiscountPercent').textContent = discount;
            document.getElementById('calcDiscountAmount').textContent = '-$' + discountAmount.toFixed(2);
            document.getElementById('calcFinalAmount').textContent = '$' + finalAmount.toFixed(2);
        }
        
        // Add event listeners for price calculation
        document.addEventListener('DOMContentLoaded', function() {
            const priceInput = document.getElementById('package_price');
            const discountInput = document.getElementById('discount_percentage');
            
            if (priceInput) {
                priceInput.addEventListener('input', calculateDealAmount);
            }
            if (discountInput) {
                discountInput.addEventListener('input', calculateDealAmount);
            }
        });
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modals = ['viewModal', 'noteModal', 'packageModal', 'priceModal'];
            modals.forEach(modalId => {
                const modal = document.getElementById(modalId);
                if (event.target === modal) {
                    if (modalId === 'viewModal') closeViewModal();
                    if (modalId === 'noteModal') closeNoteModal();
                    if (modalId === 'packageModal') closePackageModal();
                    if (modalId === 'priceModal') closePriceModal();
                }
            });
        }
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(event) {
            const modals = ['viewModal', 'noteModal', 'packageModal', 'priceModal'];
            modals.forEach(modalId => {
                const modal = document.getElementById(modalId);
                if (event.key === 'Escape' && modal.classList.contains('active')) {
                    if (modalId === 'viewModal') closeViewModal();
                    if (modalId === 'noteModal') closeNoteModal();
                    if (modalId === 'packageModal') closePackageModal();
                    if (modalId === 'priceModal') closePriceModal();
                }
            });
        });
        
        // Auto-focus on search if client_id is in URL
        <?php if (isset($_GET['client_id'])): ?>
        document.addEventListener('DOMContentLoaded', function() {
            const clientId = <?php echo intval($_GET['client_id']); ?>;
            setTimeout(() => {
                viewClient(clientId);
            }, 500);
        });
        <?php endif; ?>
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>