<?php
require_once 'config.php';
requireDepartment(['admin']);

// Date range filter
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
$department_filter = isset($_GET['department']) ? $_GET['department'] : 'all';

// Prepare SQL conditions
$date_condition = "DATE(c.created_at) BETWEEN '$start_date' AND '$end_date'";
$dept_condition = $department_filter != 'all' ? "c.current_department = '$department_filter'" : "1=1";

// Get report data
$reports = [];

// 1. Client Statistics
$reports['total_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients c WHERE $date_condition AND $dept_condition")->fetch_assoc()['count'];
$reports['paid_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients c WHERE package = 'paid' AND $date_condition AND $dept_condition")->fetch_assoc()['count'];
$reports['free_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients c WHERE package = 'free' AND $date_condition AND $dept_condition")->fetch_assoc()['count'];
$reports['completed_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients c WHERE status = 'completed' AND $date_condition AND $dept_condition")->fetch_assoc()['count'];
$reports['payment_collected'] = $conn->query("SELECT COUNT(*) as count FROM clients c WHERE payment_status = 'collected' AND $date_condition AND $dept_condition")->fetch_assoc()['count'];

// 2. Department-wise distribution
$dept_stats = $conn->query("
    SELECT current_department, COUNT(*) as count 
    FROM clients 
    WHERE $date_condition 
    GROUP BY current_department 
    ORDER BY count DESC
");

// 3. Monthly trend
$monthly_trend = $conn->query("
    SELECT 
        DATE_FORMAT(created_at, '%Y-%m') as month,
        COUNT(*) as count,
        SUM(CASE WHEN package = 'paid' THEN 1 ELSE 0 END) as paid_count,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_count
    FROM clients 
    WHERE YEAR(created_at) = YEAR(CURDATE())
    GROUP BY DATE_FORMAT(created_at, '%Y-%m')
    ORDER BY month
");

// 4. Top performing users
$top_users = $conn->query("
    SELECT 
        u.full_name,
        u.department,
        COUNT(c.id) as client_count,
        SUM(CASE WHEN c.package = 'paid' THEN 1 ELSE 0 END) as paid_count
    FROM users u
    LEFT JOIN clients c ON u.id = c.assigned_to
    WHERE $date_condition
    GROUP BY u.id
    ORDER BY client_count DESC
    LIMIT 10
");

// 5. Client growth
$growth_data = $conn->query("
    SELECT 
        DATE(created_at) as date,
        COUNT(*) as daily_count,
        SUM(COUNT(*)) OVER (ORDER BY DATE(created_at)) as running_total
    FROM clients 
    WHERE $date_condition
    GROUP BY DATE(created_at)
    ORDER BY date
");

// 6. Recent activities
$recent_activities = $conn->query("
    SELECT 
        ch.*,
        c.name as client_name,
        u.full_name as action_by_name,
        DATE_FORMAT(ch.action_date, '%Y-%m-%d %H:%i') as formatted_date
    FROM client_history ch
    JOIN clients c ON ch.client_id = c.id
    JOIN users u ON ch.action_by = u.id
    WHERE DATE(ch.action_date) BETWEEN '$start_date' AND '$end_date'
    ORDER BY ch.action_date DESC
    LIMIT 20
");

// 7. Performance metrics
$avg_processing_time = $conn->query("
    SELECT 
        AVG(DATEDIFF(
            (SELECT MAX(action_date) FROM client_history WHERE client_id = c.id AND to_department = 'completed'),
            (SELECT MIN(action_date) FROM client_history WHERE client_id = c.id)
        )) as avg_days
    FROM clients c
    WHERE c.status = 'completed' AND $date_condition
")->fetch_assoc()['avg_days'];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports & Analytics | Client Management System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .reports-header {
            background: white;
            padding: 1.5rem;
            border-radius: 10px;
            margin-bottom: 2rem;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        
        .filters {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        
        .filter-group label {
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #555;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
            text-align: center;
            transition: transform 0.3s;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-icon {
            font-size: 2rem;
            margin-bottom: 1rem;
            color: #667eea;
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: bold;
            color: #333;
            display: block;
            margin-bottom: 0.5rem;
        }
        
        .stat-label {
            font-size: 0.9rem;
            color: #666;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .chart-container {
            background: white;
            padding: 2rem;
            border-radius: 10px;
            margin-bottom: 2rem;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        
        .chart-title {
            margin-bottom: 1.5rem;
            color: #333;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .report-section {
            margin-bottom: 3rem;
        }
        
        .report-section h2 {
            margin-bottom: 1.5rem;
            color: #333;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid #f0f2f5;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }
        
        .data-table th {
            background: #f8f9fa;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: #555;
        }
        
        .data-table td {
            padding: 1rem;
            border-bottom: 1px solid #eee;
        }
        
        .data-table tr:hover {
            background: #f8f9fa;
        }
        
        .export-buttons {
            display: flex;
            gap: 1rem;
            margin-top: 2rem;
        }
        
        .btn-export {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1.5rem;
            background: #28a745;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            font-weight: 500;
            transition: all 0.3s;
        }
        
        .btn-export:hover {
            background: #218838;
            transform: translateY(-2px);
        }
        
        .btn-export.pdf {
            background: #dc3545;
        }
        
        .btn-export.pdf:hover {
            background: #c82333;
        }
        
        .trend-indicator {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.5rem;
            border-radius: 3px;
            font-size: 0.85rem;
            font-weight: 500;
        }
        
        .trend-up {
            background: #d4edda;
            color: #155724;
        }
        
        .trend-down {
            background: #f8d7da;
            color: #721c24;
        }
        
        .trend-neutral {
            background: #fff3cd;
            color: #856404;
        }
        
        .department-chart {
            height: 300px;
        }
        
        .monthly-chart {
            height: 300px;
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem;
            color: #666;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: #ddd;
        }
        
        .info-box {
            background: #e7f3ff;
            padding: 1rem;
            border-radius: 5px;
            margin: 1rem 0;
            border-left: 4px solid #1890ff;
        }
        
        .info-box p {
            margin: 0;
            color: #1890ff;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="main-content">
        <h1><i class="fas fa-chart-pie"></i> Reports & Analytics</h1>
        
        <div class="reports-header">
            <h2><i class="fas fa-filter"></i> Filter Reports</h2>
            <form method="GET" action="" class="filters">
                <div class="filter-group">
                    <label for="start_date">Start Date</label>
                    <input type="date" id="start_date" name="start_date" 
                           value="<?php echo $start_date; ?>" class="form-control">
                </div>
                
                <div class="filter-group">
                    <label for="end_date">End Date</label>
                    <input type="date" id="end_date" name="end_date" 
                           value="<?php echo $end_date; ?>" class="form-control">
                </div>
                
                <div class="filter-group">
                    <label for="department">Department</label>
                    <select id="department" name="department" class="form-control">
                        <option value="all" <?php echo $department_filter == 'all' ? 'selected' : ''; ?>>All Departments</option>
                        <option value="lead" <?php echo $department_filter == 'lead' ? 'selected' : ''; ?>>Lead</option>
                        <option value="sales" <?php echo $department_filter == 'sales' ? 'selected' : ''; ?>>Sales</option>
                        <option value="delivery" <?php echo $department_filter == 'delivery' ? 'selected' : ''; ?>>Delivery</option>
                        <option value="project" <?php echo $department_filter == 'project' ? 'selected' : ''; ?>>Project</option>
                        <option value="upfollow" <?php echo $department_filter == 'upfollow' ? 'selected' : ''; ?>>Upscale</option>
                    </select>
                </div>
                
                <div class="filter-group" style="align-self: flex-end;">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i> Generate Report
                    </button>
                    <a href="reports.php" class="btn btn-secondary" style="margin-top: 0.5rem;">
                        <i class="fas fa-redo"></i> Reset
                    </a>
                </div>
            </form>
        </div>
        
        <!-- Summary Statistics -->
        <div class="report-section">
            <h2><i class="fas fa-chart-bar"></i> Summary Statistics</h2>
            
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <span class="stat-value"><?php echo $reports['total_clients']; ?></span>
                    <span class="stat-label">Total Clients</span>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-dollar-sign"></i>
                    </div>
                    <span class="stat-value"><?php echo $reports['paid_clients']; ?></span>
                    <span class="stat-label">Paid Packages</span>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <span class="stat-value"><?php echo $reports['free_clients']; ?></span>
                    <span class="stat-label">Free Packages</span>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <span class="stat-value"><?php echo $reports['completed_clients']; ?></span>
                    <span class="stat-label">Completed</span>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-money-check-alt"></i>
                    </div>
                    <span class="stat-value"><?php echo $reports['payment_collected']; ?></span>
                    <span class="stat-label">Payments Collected</span>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-clock"></i>
                    </div>
                    <span class="stat-value"><?php echo round($avg_processing_time, 1); ?></span>
                    <span class="stat-label">Avg Processing Days</span>
                </div>
            </div>
        </div>
        
        <!-- Department Distribution Chart -->
        <div class="report-section">
            <h2><i class="fas fa-chart-pie"></i> Department Distribution</h2>
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-sitemap"></i> Clients by Department
                </div>
                <canvas id="departmentChart" class="department-chart"></canvas>
            </div>
        </div>
        
        <!-- Monthly Trend Chart -->
        <div class="report-section">
            <h2><i class="fas fa-chart-line"></i> Monthly Trends</h2>
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-calendar-alt"></i> Client Acquisition Trend
                </div>
                <canvas id="monthlyChart" class="monthly-chart"></canvas>
            </div>
        </div>
        
        <!-- Top Performing Users -->
        <div class="report-section">
            <h2><i class="fas fa-trophy"></i> Top Performing Users</h2>
            <div class="chart-container">
                <?php if ($top_users->num_rows == 0): ?>
                    <div class="empty-state">
                        <i class="fas fa-users"></i>
                        <h3>No Data Available</h3>
                        <p>No user performance data for the selected period</p>
                    </div>
                <?php else: ?>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Rank</th>
                                <th>User</th>
                                <th>Department</th>
                                <th>Total Clients</th>
                                <th>Paid Clients</th>
                                <th>Conversion Rate</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $rank = 1;
                            while($user = $top_users->fetch_assoc()): 
                                $conversion_rate = $user['client_count'] > 0 ? round(($user['paid_count'] / $user['client_count']) * 100, 1) : 0;
                            ?>
                            <tr>
                                <td>
                                    <span style="display: inline-block; width: 24px; height: 24px; background: <?php echo $rank <= 3 ? '#ffd700' : '#f0f2f5'; ?>; border-radius: 50%; text-align: center; line-height: 24px; font-weight: bold;">
                                        <?php echo $rank; ?>
                                    </span>
                                </td>
                                <td><strong><?php echo htmlspecialchars($user['full_name']); ?></strong></td>
                                <td>
                                    <span class="department-badge badge-<?php echo $user['department']; ?>" style="padding: 0.25rem 0.5rem; border-radius: 3px; font-size: 0.85rem; background: #e3f2fd; color: #1565c0;">
                                        <?php echo ucfirst($user['department']); ?>
                                    </span>
                                </td>
                                <td><?php echo $user['client_count']; ?></td>
                                <td><?php echo $user['paid_count']; ?></td>
                                <td>
                                    <span class="trend-indicator <?php echo $conversion_rate >= 50 ? 'trend-up' : ($conversion_rate >= 30 ? 'trend-neutral' : 'trend-down'); ?>">
                                        <?php echo $conversion_rate; ?>%
                                    </span>
                                </td>
                            </tr>
                            <?php $rank++; endwhile; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Recent Activities -->
        <div class="report-section">
            <h2><i class="fas fa-history"></i> Recent Activities</h2>
            <div class="chart-container">
                <?php if ($recent_activities->num_rows == 0): ?>
                    <div class="empty-state">
                        <i class="fas fa-history"></i>
                        <h3>No Recent Activities</h3>
                        <p>No activities found for the selected period</p>
                    </div>
                <?php else: ?>
                    <div style="max-height: 400px; overflow-y: auto;">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Date & Time</th>
                                    <th>Client</th>
                                    <th>Action</th>
                                    <th>From → To</th>
                                    <th>Action By</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while($activity = $recent_activities->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo $activity['formatted_date']; ?></td>
                                    <td><strong><?php echo htmlspecialchars($activity['client_name']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($activity['notes']); ?></td>
                                    <td>
                                        <span style="display: inline-flex; align-items: center; gap: 0.5rem;">
                                            <span class="department-badge" style="background: #e3f2fd; color: #1565c0; padding: 0.25rem 0.5rem; border-radius: 3px;">
                                                <?php echo ucfirst($activity['from_department']); ?>
                                            </span>
                                            <i class="fas fa-arrow-right" style="color: #666;"></i>
                                            <span class="department-badge" style="background: #e8f5e9; color: #2e7d32; padding: 0.25rem 0.5rem; border-radius: 3px;">
                                                <?php echo ucfirst($activity['to_department']); ?>
                                            </span>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($activity['action_by_name']); ?></td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Export Options -->
        <div class="report-section">
            <h2><i class="fas fa-download"></i> Export Reports</h2>
            <div class="export-buttons">
                <a href="export_reports.php?type=excel&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>&department=<?php echo $department_filter; ?>" class="btn-export">
                    <i class="fas fa-file-excel"></i> Export to Excel
                </a>
               
            </div>
        </div>
    </div>

    <script>
        // Department Distribution Chart
        <?php
        $dept_labels = [];
        $dept_data = [];
        $dept_colors = [
            'lead' => '#2196F3',
            'sales' => '#4CAF50',
            'delivery' => '#FF9800',
            'project' => '#9C27B0',
            'upfollow' => '#00BCD4',
            'completed' => '#795548'
        ];
        
        while($dept = $dept_stats->fetch_assoc()) {
            $dept_labels[] = ucfirst($dept['current_department']);
            $dept_data[] = $dept['count'];
        }
        ?>
        
        const deptCtx = document.getElementById('departmentChart').getContext('2d');
        const deptChart = new Chart(deptCtx, {
            type: 'doughnut',
            data: {
                labels: <?php echo json_encode($dept_labels); ?>,
                datasets: [{
                    data: <?php echo json_encode($dept_data); ?>,
                    backgroundColor: [
                        '#2196F3', '#4CAF50', '#FF9800', '#9C27B0', '#00BCD4', '#795548'
                    ],
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                let label = context.label || '';
                                let value = context.raw || 0;
                                let total = context.dataset.data.reduce((a, b) => a + b, 0);
                                let percentage = Math.round((value / total) * 100);
                                return `${label}: ${value} (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
        
        // Monthly Trend Chart
        <?php
        $month_labels = [];
        $month_data = [];
        $paid_data = [];
        $completed_data = [];
        
        while($month = $monthly_trend->fetch_assoc()) {
            $month_labels[] = date('M Y', strtotime($month['month'] . '-01'));
            $month_data[] = $month['count'];
            $paid_data[] = $month['paid_count'];
            $completed_data[] = $month['completed_count'];
        }
        ?>
        
        const monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
        const monthlyChart = new Chart(monthlyCtx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($month_labels); ?>,
                datasets: [
                    {
                        label: 'Total Clients',
                        data: <?php echo json_encode($month_data); ?>,
                        borderColor: '#667eea',
                        backgroundColor: 'rgba(102, 126, 234, 0.1)',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4
                    },
                    {
                        label: 'Paid Clients',
                        data: <?php echo json_encode($paid_data); ?>,
                        borderColor: '#10b981',
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        borderWidth: 2,
                        fill: false,
                        tension: 0.4
                    },
                    {
                        label: 'Completed',
                        data: <?php echo json_encode($completed_data); ?>,
                        borderColor: '#ef4444',
                        backgroundColor: 'rgba(239, 68, 68, 0.1)',
                        borderWidth: 2,
                        fill: false,
                        tension: 0.4
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Number of Clients'
                        }
                    }
                }
            }
        });
        
        // Auto-update charts on window resize
        window.addEventListener('resize', function() {
            deptChart.resize();
            monthlyChart.resize();
        });
        
        // Print functionality
        function printReport() {
            window.print();
        }
        
        // Date range validation
        document.querySelector('form').addEventListener('submit', function(e) {
            const startDate = new Date(document.getElementById('start_date').value);
            const endDate = new Date(document.getElementById('end_date').value);
            
            if (startDate > endDate) {
                e.preventDefault();
                alert('Start date cannot be after end date!');
                return false;
            }
        });
    </script>
</body>
</html>