<?php
require_once 'config.php';
requireDepartment(['project', 'admin']);

// Handle client updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_client'])) {
    $client_id = intval($_POST['client_id']);
    $status = $conn->real_escape_string($_POST['status']);
    $payment_status = $conn->real_escape_string($_POST['payment_status']);
    $note = $conn->real_escape_string($_POST['note']);
    $transfer_to = isset($_POST['transfer_to']) ? $conn->real_escape_string($_POST['transfer_to']) : null;
    
    // Update client
    $stmt = $conn->prepare("UPDATE clients SET status = ?, payment_status = ?, note = CONCAT(COALESCE(note, ''), '\nProject Update: ', ?), updated_at = NOW() WHERE id = ?");
    $stmt->bind_param("sssi", $status, $payment_status, $note, $client_id);
    $stmt->execute();
    
    // Transfer if requested
    if ($transfer_to && $transfer_to !== 'none') {
        $update_stmt = $conn->prepare("UPDATE clients SET current_department = ? WHERE id = ?");
        $update_stmt->bind_param("si", $transfer_to, $client_id);
        $update_stmt->execute();
        
        // Log the transfer
        $action_stmt = $conn->prepare("INSERT INTO client_history (client_id, from_department, to_department, action_by, notes) VALUES (?, 'project', ?, ?, ?)");
        $action_notes = "Project update - Status: $status, Payment: $payment_status. " . $note;
        $action_stmt->bind_param("isis", $client_id, $transfer_to, $_SESSION['user_id'], $action_notes);
        $action_stmt->execute();
    }
    
    header('Location: project_dept.php?success=1');
    exit();
}

// Get clients in project department
$clients = $conn->query("SELECT * FROM clients WHERE current_department = 'project' ORDER BY created_at DESC");
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Project Management | Client Management System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="main-content">
        <h1><i class="fas fa-project-diagram"></i> Project Management Department</h1>
        
        <?php if (isset($_GET['success'])): ?>
        <div class="alert alert-success">
            Client updated successfully!
        </div>
        <?php endif; ?>
        
        <div class="table-container">
            <h2><i class="fas fa-users"></i> Clients in Project Department</h2>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Phone</th>
                        <th>Package</th>
                        <th>Status</th>
                        <th>Payment</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php while($client = $clients->fetch_assoc()): ?>
                    <tr>
                        <td>#<?php echo str_pad($client['id'], 5, '0', STR_PAD_LEFT); ?></td>
                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                        <td><?php echo htmlspecialchars($client['phone']); ?></td>
                        <td>
                            <span class="status-badge <?php echo $client['package'] === 'paid' ? 'status-completed' : 'status-pending'; ?>">
                                <?php echo ucfirst($client['package']); ?>
                            </span>
                        </td>
                        <td>
                            <span class="status-badge status-<?php echo str_replace('_', '-', $client['status']); ?>">
                                <?php echo ucwords(str_replace('_', ' ', $client['status'])); ?>
                            </span>
                        </td>
                        <td>
                            <span class="status-badge <?php echo $client['payment_status'] === 'collected' ? 'status-completed' : 'status-pending'; ?>">
                                <?php echo ucfirst($client['payment_status']); ?>
                            </span>
                        </td>
                        <td>
                            <button class="btn btn-primary btn-sm" onclick="openProjectModal(<?php echo $client['id']; ?>)">
                                <i class="fas fa-edit"></i> Update
                            </button>
                        </td>
                    </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Project Update Modal -->
    <div class="modal" id="projectModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Update Client Project Status</h3>
                <button class="btn-close" onclick="closeProjectModal()">&times;</button>
            </div>
            <form id="projectForm" method="POST">
                <input type="hidden" name="client_id" id="client_id">
                <div class="form-group">
                    <label for="status">Project Status</label>
                    <select id="status" name="status" class="form-control" required>
                        <option value="pending">Pending</option>
                        <option value="in_progress">In Progress</option>
                        <option value="completed">Completed</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="payment_status">Payment Status</label>
                    <select id="payment_status" name="payment_status" class="form-control" required>
                        <option value="pending">Pending</option>
                        <option value="collected">Collected</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="transfer_to">Transfer To Department</label>
                    <select id="transfer_to" name="transfer_to" class="form-control">
                        <option value="none">Keep in Project Department</option>
                        <option value="upfollow">Upscale Department</option>
                        <option value="delivery">Back to Delivery</option>
                        <option value="sales">Back to Sales</option>
                        <option value="lead">Back to Lead</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="note">Update Note</label>
                    <textarea id="note" name="note" class="form-control" rows="3" required></textarea>
                </div>
                <button type="submit" name="update_client" class="btn btn-primary">
                    <i class="fas fa-save"></i> Update Client
                </button>
            </form>
        </div>
    </div>

    <script>
        function openProjectModal(clientId) {
            document.getElementById('client_id').value = clientId;
            document.getElementById('projectModal').classList.add('active');
        }
        
        function closeProjectModal() {
            document.getElementById('projectModal').classList.remove('active');
        }
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>